"use client";
import { useEffect, useId, useState } from "react";
import {
  DndContext,
  closestCenter,
  KeyboardSensor,
  PointerSensor,
  useSensor,
  useSensors,
} from "@dnd-kit/core";
import {
  arrayMove,
  SortableContext,
  sortableKeyboardCoordinates,
  verticalListSortingStrategy,
} from "@dnd-kit/sortable";
import {
  Card,
  CardContent,
  CardHeader,
  CardTitle,
  CardDescription,
} from "@/components/ui/card";
import { Check, Loader2, Plus } from "lucide-react";
import { Button } from "@/components/ui/button";
import Content from "@/components/dashboard/content";
import SortableBioItem from "@/components/biolink/bio-item";
import PhoneMockup from "@/components/biolink/phone-mockup";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import ButtonPresets from "@/components/biolink/button-presets";
import ButtonCustomizer from "@/components/biolink/button-customizer";
import ThemePresets from "@/components/biolink/theme-presets";
import ProfileSection from "@/components/biolink/profile-section";
import ThemeCustomizer from "@/components/biolink/theme-customizer";
import { useUpdateBiolink, useViewBiolink } from "@/services/biolink-services";
import {
  BiolinkBackgroundDesignDTO,
  BiolinkButtonDesignDTO,
  BiolinkDesignDTO,
  BiolinkDTO,
  LinkDTO,
} from "@/DTO/biolink";
import ProfileInput from "@/components/biolink/profile-input";
import BackButton from "@/components/shared/buttons/back-button";
import Image from "next/image";
import ErrorAlert from "@/components/shared/alert/error-alert";
import { useCustomClaims } from "@/services/user-services";
import { toast } from "sonner";

export default function BioLinkUpdate({ id }) {
  const { userSubscription } = useCustomClaims();
  const uniqueId = useId();
  const {
    biolink,
    biolinkDesign,
    isLoading,
    isValidating,
    mutate,
    success,
    message,
  } = useViewBiolink(id);
  const { trigger, isMutating } = useUpdateBiolink();
  const [items, setItems] = useState([]);
  const [itemId, setItemId] = useState(1);
  const [itemsSet, setItemsSet] = useState(false); // Flag to ensure items are only set once
  const [profileInfo, setProfileInfo] = useState({
    name: "",
    bio: "",
    profilePicture: "",
  });
  const [buttonDesign, setButtonDesign] = useState({
    name: "Deafult",
    design: {
      type: "solid",
      height: 0.5,
      backgroundColor: "#ffffff",
      borderWidth: 0.125,
      borderRadius: 0.5,
      borderColor: "#e2e8f0",
      textColor: "#000000",
      shadow: "",
    },
  });
  const [theme, setTheme] = useState({
    name: "Minimalist",
    themeDesign: {
      backgroundType: "solid",
      backgroundImage: "",
      backgroundColor: "#f5f5f4",
      backgroundGradient: "",
    },
    buttonDesign: {
      type: "solid",
      height: 0.5,
      backgroundColor: "#ffffff",
      borderWidth: 0.125,
      borderRadius: 0.5,
      borderColor: "#e2e8f0",
      textColor: "#000000",
      shadow: "",
    },
  });

  // Update items when biolink data is fetched from the server
  useEffect(() => {
    if (!isLoading && !isValidating && biolink && !itemsSet) {
      // const linkDTO1 = biolink.links.map((link) => {
      //   return new LinkDTO()
      //     .setId(link?.id)
      //     .setBiolink(link?.biolink)
      //     .setTitle(link?.title)
      //     .setDescription(link?.description)
      //     .setUrl(link?.url)
      //     .setImage(link?.image)
      //     .setDesign(link?.design)
      //     .setLayout(link?.layout)
      //     .setSchedule(link?.schedule)
      //     .setProtected(link?.protected)
      //     .build();
      // });
      // console.log(biolink);
      // if (biolink?.links.length === 0) {
      //   setItemsSet(true);
      //   return;
      // }
      if (biolink?.links) {
        const linkDTO = biolink?.links.map((link, index) => {
          return {
            id: index + 1,
            blockId: link?.id,
            biolink: link?.biolink,
            title: link?.title,
            description: link?.description,
            link: link?.url,
            image: link?.image,
            design: link?.design,
            layout: link?.layout,
            schedule: link?.schedule,
            protected: link?.protected,
          };
        });

        const profileInfo = {
          name: biolink.name,
          bio: biolink.bio,
          profilePicture: biolink.profilePicture,
        };

        setItemId(biolink.links.length + 1 || 1);
        setItems(linkDTO);
        setItemsSet(true);
        setProfileInfo(profileInfo);
        setButtonDesign({
          ...buttonDesign,
          design: biolinkDesign?.buttonDesign,
        });
        setTheme({
          ...theme,
          themeDesign: biolinkDesign?.backgroundDesign,
        });
      }
    }
  }, [
    biolink,
    isLoading,
    isValidating,
    itemsSet,
    itemId,
    biolinkDesign,
    theme,
    buttonDesign,
  ]);

  const sensors = useSensors(
    useSensor(PointerSensor),
    useSensor(KeyboardSensor, {
      coordinateGetter: sortableKeyboardCoordinates,
    })
  );

  function handleDragEnd(event) {
    const { active, over } = event;

    if (active.id !== over.id) {
      setItems((items) => {
        const oldIndex = items.findIndex((item) => item.id === active.id);
        const newIndex = items.findIndex((item) => item.id === over.id);

        return arrayMove(items, oldIndex, newIndex);
      });
    }
  }

  // Return error message if the request is not successful
  if (success == false) {
    return (
      <Content
        title={`Error`}
        extra={<BackButton />}
        isValidating={isValidating}
      >
        <ErrorAlert title={`Error`} message={message} />
      </Content>
    );
  }

  function handleAddNewBlock() {
    const userPlan = userSubscription?.plan;
    const itemLength = items?.length;
    // Restrict free user for up to 20 links
    if (userPlan == "free" && itemLength >= 20) {
      toast.warning("Free users are allowed to create up to 20 blocks.");
      return;
    }

    // Restrict Pro user for up to 40 links
    if (itemLength >= 40) {
      toast.warning("Up to 40 blocks are allowed.");
      return;
    }

    setItemId((id) => id + 1);

    setItems((items) => [
      ...items,
      {
        id: itemId,
        blockId: "",
        type: "button",
        title: "New Block",
        link: "",
        design: "primary", // default, highlight etc
        layout: "classic", // classic, block etc
        schedule: { enabled: false, start: "", end: "" },
        protected: { enabled: false, password: "" },
      },
    ]);
  }

  function chnageButtonDesign(newDesign) {
    setButtonDesign((buttonDesign) => {
      return {
        ...buttonDesign,
        ...newDesign,
      };
    });
  }

  function changeThemeDesign(newThemeDesign, newButtonDesign) {
    setTheme((themeDesign) => {
      return {
        themeDesign: {
          ...themeDesign.themeDesign,
          ...newThemeDesign,
        },
        buttonDesign: {
          ...themeDesign.buttonDesign,
          ...newButtonDesign,
        },
      };
    });
    setButtonDesign((buttonDesign) => {
      return {
        ...buttonDesign,
        design: { ...buttonDesign.design, ...newButtonDesign },
      };
    });
  }

  function changeThemeBackground(themeDesign) {
    setTheme((theme) => ({
      ...theme,
      themeDesign: {
        ...theme.themeDesign,
        ...themeDesign,
      },
    }));
  }

  function handleDeleteItem(item) {
    setItems((items) => items.filter((i) => i !== item));
  }

  function handleChangeItemTitle(item, title) {
    item.title = title;
    setItems([...items]);
  }

  function handleChangeItemLink(item, link) {
    item.link = link;
    setItems([...items]);
  }

  function handleProfileNameChange(name) {
    profileInfo.name = name;
    setProfileInfo({ ...profileInfo });
  }

  function handleProfileBioChange(bio) {
    setProfileInfo({ ...profileInfo, bio });
  }

  function handleProfilePictureChange(profilePicture) {
    setProfileInfo({ ...profileInfo, profilePicture });
  }

  const buttonDesignConfig = buttonDesign?.design;
  const themeConfig = theme?.themeDesign;

  async function handleUpdateBiolink() {
    const linkDTO = items.map((item) => {
      return new LinkDTO()
        .setId(item?.blockId)
        .setBiolink(item?.biolink)
        .setTitle(item?.title)
        .setDescription(item?.description)
        .setUrl(item?.link)
        .setImage(item?.image)
        .build();
    });

    const biolinkDTO = new BiolinkDTO()
      .setId(biolink?.id)
      .setName(profileInfo?.name)
      .setProfilePicture(profileInfo?.profilePicture?.id)
      .setBio(profileInfo?.bio)
      .setLinks(linkDTO)
      .build();

    const biolinkBackgroundDesignDTO = new BiolinkBackgroundDesignDTO()
      .setBackgroundType(themeConfig?.backgroundType)
      .setBackgroundImage(themeConfig?.backgroundImage?.id)
      .setBackgroundColor(themeConfig?.backgroundColor)
      .setBackgroundGradient(themeConfig?.backgroundGradient)
      .build();

    const biolinkButtonDesignDTO = new BiolinkButtonDesignDTO()
      .setType(buttonDesignConfig?.type)
      .setHeight(buttonDesignConfig?.height)
      .setBackgroundColor(buttonDesignConfig?.backgroundColor)
      .setBorderWidth(buttonDesignConfig?.borderWidth)
      .setBorderRadius(buttonDesignConfig?.borderRadius)
      .setBorderColor(buttonDesignConfig?.borderColor)
      .setTextColor(buttonDesignConfig?.textColor)
      .setShadow(buttonDesignConfig?.shadow)
      .build();

    const biolinkDesignDTO = new BiolinkDesignDTO()
      .setBiolink(biolink?.id)
      .setBackgroundDesign(biolinkBackgroundDesignDTO)
      .setButtonDesign(biolinkButtonDesignDTO)
      .build();

    await trigger({
      biolink: biolinkDTO,
      design: biolinkDesignDTO,
    });

    await mutate();
  }

  function SaveBioLinkButton() {
    return (
      <div className="flex gap-2 flex-row items-center">
        <BackButton />
        <Button
          className="gap-1"
          onClick={handleUpdateBiolink}
          disabled={isMutating || isLoading}
          size="sm"
        >
          {isMutating ? (
            <Loader2 className="h-4 w-4 animate-spin" />
          ) : (
            <Check className="h-4 w-4" />
          )}
          <span>Update</span>
        </Button>
      </div>
    );
  }

  return (
    <Content
      title={biolink?.username}
      extra={<SaveBioLinkButton />}
      isValidating={isValidating}
    >
      <style jsx>
        {`
          .block_button {
            padding: ${buttonDesignConfig?.height}rem 0;
            border: ${buttonDesignConfig?.borderWidth}rem solid
              ${buttonDesignConfig?.borderColor};
            border-radius: ${buttonDesignConfig?.borderRadius}rem;
            color: ${buttonDesignConfig?.textColor};
            box-shadow: ${buttonDesignConfig?.shadow};
            ${buttonDesignConfig?.type == "solid"
              ? `background-color: ${buttonDesignConfig?.backgroundColor}`
              : `background-image: ${buttonDesignConfig?.backgroundColor}`};
          }
        `}
      </style>
      <style jsx>
        {`
          .theme {
            ${themeConfig?.backgroundType == "image"
              ? `
                background-image: url(${themeConfig?.backgroundImage?.url});
                background-size: cover;
                background-position: center;
                `
              : ""}

            ${themeConfig?.backgroundType == "gradient"
              ? `background-image: ${themeConfig?.backgroundGradient};`
              : ""}

            ${themeConfig?.backgroundType == "solid"
              ? `background-color: ${themeConfig?.backgroundColor};`
              : ""}
          }
        `}
      </style>
      <style jsx>
        {`
          .background-container {
            position: relative;
            width: 100%;
            height: 100%;
            overflow: hidden;
          }
        `}
      </style>
      <style jsx>
        {`
          .image-container {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            overflow: hidden;
            z-index: 1;
          }
        `}
      </style>
      <style jsx>
        {`
          .content-container {
            position: relative;
            top: 50%;
            left: 50%;
            z-index: 100;
            transform: translate(-50%, -50%);
            color: red;
          }
        `}
      </style>
      <div className="grid gap-4 lg:grid-cols-2 xl:grid-cols-2">
        {/* Left side. Customize Bio link */}
        <div className="col-span-1">
          <Card className="h-full">
            <CardHeader>
              <div className="grid gap-2">
                <CardTitle>Customize bio link</CardTitle>
                <CardDescription>
                  Edit your bio link and track its performance
                </CardDescription>
              </div>
            </CardHeader>
            <CardContent>
              <Tabs defaultValue="block">
                <TabsList className="w-full mb-6">
                  <TabsTrigger value="block" className="w-full">
                    Blocks
                  </TabsTrigger>
                  <TabsTrigger value="profile" className="w-full">
                    Profile
                  </TabsTrigger>
                  <TabsTrigger value="design" className="w-full">
                    Design
                  </TabsTrigger>
                </TabsList>
                <TabsContent
                  forceMount
                  className="data-[state=inactive]:hidden"
                  value="block"
                >
                  {/* Bio link drag and drop link builder */}
                  <div className="grid gap-4">
                    <Button
                      variant="outline"
                      className="w-full border-primary"
                      onClick={handleAddNewBlock}
                      disabled={isLoading}
                    >
                      <Plus className="h-4 w-4" />
                      Add new block
                    </Button>
                    <DndContext
                      sensors={sensors}
                      collisionDetection={closestCenter}
                      onDragEnd={handleDragEnd}
                      id={uniqueId}
                    >
                      <SortableContext
                        items={items}
                        strategy={verticalListSortingStrategy}
                      >
                        {items?.map((item) => (
                          <SortableBioItem
                            key={item?.id}
                            item={item}
                            handleDeleteItem={handleDeleteItem}
                            handleChangeItemTitle={handleChangeItemTitle}
                            handleChangeItemLink={handleChangeItemLink}
                          />
                        ))}
                      </SortableContext>
                    </DndContext>
                  </div>
                </TabsContent>
                <TabsContent
                  forceMount
                  className="data-[state=inactive]:hidden"
                  value="profile"
                >
                  <ProfileInput
                    currentProfileInfo={profileInfo}
                    handleProfileNameChange={handleProfileNameChange}
                    handleProfileBioChange={handleProfileBioChange}
                    handleProfilePictureChange={handleProfilePictureChange}
                  />
                </TabsContent>
                <TabsContent
                  forceMount
                  className="data-[state=inactive]:hidden"
                  value="design"
                >
                  {/* Design */}
                  <div>
                    <Tabs defaultValue="theme" className="w-full">
                      <TabsList className="mb-4 w-full">
                        <TabsTrigger value="theme" className="w-full">
                          Theme
                        </TabsTrigger>
                        <TabsTrigger value="button" className="w-full">
                          Button
                        </TabsTrigger>
                        <TabsTrigger value="custom" className="w-full">
                          Custom
                        </TabsTrigger>
                      </TabsList>
                      <TabsContent
                        forceMount
                        className="data-[state=inactive]:hidden"
                        value="theme"
                      >
                        <ThemePresets changeThemeDesign={changeThemeDesign} />
                      </TabsContent>
                      <TabsContent
                        forceMount
                        className="data-[state=inactive]:hidden"
                        value="button"
                      >
                        <div className="grid gap-4">
                          <strong>Presets</strong>
                          {/* Button Presets */}
                          <ButtonPresets
                            chnageButtonDesign={chnageButtonDesign}
                          />
                        </div>
                      </TabsContent>
                      <TabsContent
                        forceMount
                        className="data-[state=inactive]:hidden"
                        value="custom"
                      >
                        <div className="grid gap-12">
                          <div className="grid gap-4">
                            {/* Button customizer ⚙️🔧 */}
                            <strong>Customize Button</strong>
                            <ButtonCustomizer
                              chnageButtonDesign={chnageButtonDesign}
                              currentDesign={buttonDesign}
                            />
                          </div>
                          <div className="grid gap-4">
                            {/* Theme customizer ⚙️🔧 */}
                            <strong>Customize Background</strong>
                            <ThemeCustomizer
                              changeDesign={changeThemeBackground}
                              currentDesign={theme}
                            />
                          </div>
                        </div>
                      </TabsContent>
                    </Tabs>
                  </div>
                  {/* Design end */}
                </TabsContent>
              </Tabs>
            </CardContent>
          </Card>
        </div>
        {/* Right side. Preview 2️⃣*/}
        <div className="col-span-1">
          <Card className="h-full">
            <CardHeader>
              <CardTitle>Preview your bio link</CardTitle>
              <CardDescription>
                Preview your bio link as your audience will see
              </CardDescription>
            </CardHeader>
            <CardContent className="h-full">
              <div className="h-full min-h-[680px]">
                {/* Bio link preview */}
                {/* Render buttons based on the length of items */}
                <div className="sticky top-0">
                  {/* Preview inside phone mockup */}
                  <PhoneMockup>
                    <div className="relative w-full h-full overflow-hidden">
                      {/* Image */}
                      <div className="absolute top-0 left-0 w-full h-full overflow-hidden">
                        {
                          // If background type is image, show the image
                          themeConfig?.backgroundType === "image" && (
                            <Image
                              src={themeConfig?.backgroundImage?.url}
                              alt={themeConfig?.backgroundImage?.alt}
                              width={300}
                              height={600}
                              unoptimized
                              className="object-cover w-auto h-full"
                            />
                          )
                        }
                        {
                          // If background type is gradient, show the gradient
                          themeConfig?.backgroundType === "gradient" && (
                            <div className="theme w-full h-full"></div>
                          )
                        }
                        {
                          // If background type is solid, show the solid color
                          themeConfig?.backgroundType === "solid" && (
                            <div className="theme w-full h-full"></div>
                          )
                        }
                      </div>
                      {/* Content */}
                      <div className="relative h-full">
                        <div className="flex flex-col gap-4 w-full p-4 bg-transparent h-full overflow-y-auto [&::-webkit-scrollbar]:hidden [-ms-overflow-style:none] [scrollbar-width:none]">
                          {/* Profile Section */}
                          <ProfileSection profileInfo={profileInfo} />

                          {/* Button Maps */}
                          {items?.map((item) => (
                            <button
                              className={"block_button"}
                              key={item?.id}
                              title={item?.link}
                            >
                              {item?.title}
                            </button>
                          ))}

                          {/* If no block is added, ask for a block to add */}
                          {items?.length == 0 && (
                            <div className="text-center">
                              <p className="text-sm">
                                Add a new block to your bio link
                              </p>
                            </div>
                          )}
                        </div>
                      </div>
                    </div>
                  </PhoneMockup>
                </div>
              </div>
            </CardContent>
          </Card>
        </div>
      </div>
    </Content>
  );
}
